#include "Scene.h"
#include "FreeCamera.h"
#include "StrafeCamera.h"
#include "SelfRotatingCamera.h"

#include "Shape.h"
#include "Sphere.h"
#include "Torus.h"

#include "Model.h"
#include "PoolTable.h"
#include "Booth.h"
#include "SlotMachine.h"
#include "Jukebox.h"
#include "Phone.h"

#include "LightManager.h"
#include "SpotLight.h"
#include "PointLight.h"

#include <string>
#include <memory>

#pragma region Cameras
FreeCamera freeCamera(1.0f, Vector3(1.0f, 3.5f, -2.5f), Vector3(-15.0f, 35.0f, 0.0f), Vector3(0.5f, 0.5f, -1.0f), Vector3(12.0f, 3.5f, -14.5f));	//Free camera, in the top left corner, aimed toward pool table
StrafeCamera barStrafeCamera(Vector3(8.0f, 1.5f, -5.0f), Vector3(0.0f, 90.0f, 0.0f), StrafeCamera::StrafeAxis::Z, -1.5f, -14.5f);	//Strafe camera, pointed to and strafes up and down the bar
SelfRotatingCamera securityCamera(Vector3(12.5f, 3.5f, -15.5f), Vector3(-25.0f, 220.0f, 0.0f), 0.75f, SelfRotatingCamera::RotationAxis::Y, 205.0f, 235.0f);	//"Security" camera in the back right corner of the bar
Camera* sceneCamera;	//Pointer to active camera
#pragma endregion

#pragma region Textures
GLuint floorTileTexture;
GLuint roofTileTexture;
GLuint wallTexture;
GLuint crateTexture;
GLuint windowTexture;
GLuint matTexture;
GLuint duffTallTexture;
GLuint duffWideTexture;
GLuint chalkboardTexture;
GLuint doorMatTexture;
GLuint doorTexture;
GLuint donutTexture;
GLuint flagTexture;
#pragma endregion

#pragma region Models

PoolTable poolTable;
Booth booth;
Model bar;
Model seat;
Model shelf;
Model phone;
Jukebox jukebox;
Model barrel;
SlotMachine slotMachine;
Torus donut;
Model colaCan;
Model glassTable;

#pragma endregion

#pragma region Lighting
LightManager lightManager;	//Manager for all our lights

//Pool table spot lights setup
std::vector<GLfloat> PoolSpotPositionOne = { 6.0f, 3.75f, -10.5f, 1.0f };
std::vector<GLfloat> PoolSpotPositionTwo = { 6.0f, 3.75f, -8.0f, 1.0f };
std::vector<GLfloat> PoolSpotDirection = { 0.0f, -1.0f, 0.0f };

//Slot machine spot lights setup
std::vector<GLfloat> WinSpotPosition = { 14.0f, 2.0f, -2.0f, 1.0f };
std::vector<GLfloat> WinSpotDirection = { -1.0f, 0.0f, 0.0f };

//Pool table spot light defintions
SpotLight poolSpotOne = SpotLight(GL_LIGHT4, PoolSpotPositionOne, PoolSpotDirection, 75.0f, 5.0f);
SpotLight poolSpotTwo = SpotLight(GL_LIGHT5, PoolSpotPositionTwo, PoolSpotDirection, 75.0f, 5.0f);
SpotLight slotGreenWinSpot = SpotLight(GL_LIGHT6, WinSpotPosition, WinSpotDirection, 30.0f, 15.0f);

//Roof point light setup and definitions
std::vector<GLfloat> PositionOne = { 3.0f, 3.0f, -14.0f, 1.0f };
PointLight pointLightOne = PointLight(GL_LIGHT0, PositionOne);
std::vector<GLfloat> PositionTwo = { 10.0f, 3.0f, -14.0f, 1.0f };
PointLight pointLightTwo = PointLight(GL_LIGHT1, PositionTwo);
std::vector<GLfloat> PositionThree = { 3.0f, 3.0f, -2.0f, 1.0f };
PointLight pointLightThree = PointLight(GL_LIGHT2, PositionThree);
std::vector<GLfloat> PositionFour = { 10.0f, 3.0f, -2.0f, 1.0f };
PointLight pointLightFour = PointLight(GL_LIGHT3, PositionFour);
#pragma endregion

bool wireframeOn = false;

//Returns a loaded texture from the path
GLuint LoadTexture(std::string path)
{
	return SOIL_load_OGL_texture
	(
		path.c_str(),
		SOIL_LOAD_AUTO,
		SOIL_CREATE_NEW_ID,
		SOIL_FLAG_MIPMAPS | SOIL_FLAG_NTSC_SAFE_RGB | SOIL_FLAG_COMPRESS_TO_DXT
	);
}

Scene::Scene(Input *in)
{
	// Store pointer for input class
	input = in;
	glutWarpPointer((glutGet(GLUT_WINDOW_WIDTH) / 2), (glutGet(GLUT_WINDOW_HEIGHT) / 2));

	#pragma region OpenGL Initial Settings
	glShadeModel(GL_SMOOTH);							// Enable Smooth Shading
	glClearColor(0.39f, 0.58f, 93.0f, 1.0f);			// Cornflour Blue Background
	glClearDepth(1.0f);									// Depth Buffer Setup
	glClearStencil(0);									// Clear stencil buffer
	glEnable(GL_DEPTH_TEST);							// Enables Depth Testing
	glDepthFunc(GL_LEQUAL);								// The Type Of Depth Testing To Do
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);	// Really Nice Perspective Calculations
	glutSetCursor(GLUT_CURSOR_NONE);
	glEnable(GL_LIGHTING);
	glEnable(GL_TEXTURE_2D);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	#pragma endregion

	#pragma region Texture Loading
	floorTileTexture = LoadTexture("Assets/Textures/floorTiles.png");
	roofTileTexture = LoadTexture("Assets/Textures/roofTiles.png");
	wallTexture = LoadTexture("Assets/Textures/wall.png");
	crateTexture = LoadTexture("Assets/Textures/crate.png");
	windowTexture = LoadTexture("Assets/Textures/window.png");
	matTexture = LoadTexture("Assets/Textures/mat.png");
	duffTallTexture = LoadTexture("Assets/Textures/duffPosterTall.png");
	duffWideTexture = LoadTexture("Assets/Textures/duffPosterWide.png");
	chalkboardTexture = LoadTexture("Assets/Textures/chalkboard.png");
	doorMatTexture = LoadTexture("Assets/Textures/mat.png");
	doorTexture = LoadTexture("Assets/Textures/door.png");
	donutTexture = LoadTexture("Assets/Textures/donut.png");
	flagTexture = LoadTexture("Assets/Textures/flag.png");
	#pragma endregion

	sceneCamera = &freeCamera;	//Default camera is the free camera

	#pragma region Model Loading
	poolTable.loadMTL("Assets/Models/poolTable.obj", "Assets/Models/poolTable.mtl");
	booth.loadMTL("Assets/Models/booth.obj", "Assets/Models/booth.mtl");
	bar.loadMTL("Assets/Models/bar.obj", "Assets/Models/bar.mtl");
	shelf.loadMTL("Assets/Models/barShelf.obj", "Assets/Models/barShelf.mtl");
	seat.loadMTL("Assets/Models/barStool.obj", "Assets/Models/barStool.mtl");
	slotMachine.loadMTL("Assets/Models/slotMachine.obj", "Assets/Models/slotMachine.mtl");
	phone.loadMTL("Assets/Models/phone.obj", "Assets/Models/phone.mtl");
	jukebox.loadMTL("Assets/Models/jukebox.obj", "Assets/Models/jukebox.mtl");
	barrel.loadMTL("Assets/Models/barrel.obj", "Assets/Models/barrel.mtl");
	colaCan.loadMTL("Assets/Models/cola.obj", "Assets/Models/cola.mtl");
	glassTable.loadMTL("Assets/Models/table.obj", "Assets/Models/table.mtl");
	#pragma endregion

	donut.Generate(donutTexture, 25, 0.07f, 0.04f);	//Generate the donut for the bar

	#pragma region Light Setup

	//Attenuation Setup
	pointLightOne.SetAttenuation(0.8f, 0.1f, 0.01f);
	pointLightTwo.SetAttenuation(0.8f, 0.1f, 0.01f);
	pointLightThree.SetAttenuation(0.8f, 0.1f, 0.01f);
	pointLightFour.SetAttenuation(0.8f, 0.1f, 0.01f);
	poolSpotOne.SetAttenuation(0.4f, 0.05f, 0.0075f);
	poolSpotTwo.SetAttenuation(0.4f, 0.05f, 0.0075f);
	slotGreenWinSpot.SetAttenuation(0.01f, 0.01f, 0.0075f);

	//Slot machine light
	slotGreenWinSpot.SetAmbient(0.0f, 1.0f, 0.0f);
	slotGreenWinSpot.SetDiffuse(0.0f, 1.0f, 0.0f);
	
	//Light Manager
	lightManager.AddNewLight(&pointLightOne);
	lightManager.AddNewLight(&pointLightTwo);
	lightManager.AddNewLight(&pointLightThree);
	lightManager.AddNewLight(&pointLightFour);
	lightManager.AddNewLight(&poolSpotOne);
	lightManager.AddNewLight(&poolSpotTwo);
	lightManager.AddNewLight(&slotGreenWinSpot);

	//Shadow lights setup
	slotMachine.SetLights(&slotGreenWinSpot);
	jukebox.SetLights(&pointLightOne, &pointLightTwo);
	poolTable.setSpotlights(&poolSpotOne, &poolSpotTwo);

	//Turn on all lights
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);
	glEnable(GL_LIGHT2);
	glEnable(GL_LIGHT3);
	glEnable(GL_LIGHT4);
	glEnable(GL_LIGHT5);
	#pragma endregion
}

void Scene::handleInput(float dt)
{
	#pragma region Camera Controls
	if (input->isKeyDown('w'))
	{
		sceneCamera->MoveForward(dt);
	}

	if (input->isKeyDown('s'))
	{
		sceneCamera->MoveForward(-dt);
	}

	if (input->isKeyDown('a'))
	{
		sceneCamera->MoveRight(-dt);
	}

	if (input->isKeyDown('d'))
	{
		sceneCamera->MoveRight(dt);
	}

	if (input->isKeyDown('i'))
	{
		sceneCamera->MoveUp(dt);
	}

	if (input->isKeyDown('k'))
	{
		sceneCamera->MoveUp(-dt);
	}

	if (input->isKeyDown('t'))
	{
		sceneCamera->RotateX(dt);
	}

	if (input->isKeyDown('g'))
	{
		sceneCamera->RotateX(-dt);
	}

	if (input->isKeyDown('y'))
	{
		sceneCamera->RotateY(dt);
	}

	if (input->isKeyDown('h'))
	{
		sceneCamera->RotateY(-dt);
	}

	if (input->isKeyDown('u'))
	{
		sceneCamera->RotateZ(dt);
	}

	if (input->isKeyDown('j'))
	{
		sceneCamera->RotateZ(-dt);
	}

	if (input->isKeyDown('r'))
	{
		sceneCamera->Reset();
	}
	#pragma endregion

	//Wireframe mode
	if (input->isKeyDown('#'))
	{
		if (wireframeOn)
		{
			glPolygonMode(GL_FRONT, GL_FILL);
		}
		else if (!wireframeOn)
		{
			glPolygonMode(GL_FRONT, GL_LINE);
		}
		wireframeOn = !wireframeOn;

		input->SetKeyUp('#');
	}

	#pragma region Camera Switching
	//Switch to free camera
	if (input->isKeyDown('1'))
	{
		sceneCamera = &freeCamera;
		input->SetKeyUp('1');
	}

	//Switch to strafe camera
	if (input->isKeyDown('2'))
	{
		sceneCamera = &barStrafeCamera;
		input->SetKeyUp('2');
	}

	//Switch to rotating camera
	if (input->isKeyDown('3'))
	{
		sceneCamera = &securityCamera;
		input->SetKeyUp('3');
	}
	#pragma endregion

	#pragma region Light Toggling
	if (input->isKeyDown('4'))
	{
		lightManager.ToggleLight(&pointLightOne);
		lightManager.ToggleLight(&pointLightThree);
		input->SetKeyUp('4');
	}

	if (input->isKeyDown('5'))
	{
		lightManager.ToggleLight(&pointLightTwo);
		lightManager.ToggleLight(&pointLightFour);
		input->SetKeyUp('5');
	}

	if (input->isKeyDown('6'))
	{
		lightManager.ToggleLight(&poolSpotOne);
		lightManager.ToggleLight(&poolSpotTwo);
		input->SetKeyUp('6');
	}
	#pragma endregion

	//Slot machine
	if (input->isKeyDown('0'))
	{
		slotMachine.Activate();
		input->SetKeyUp('0');
	}
}

void Scene::update(float dt)
{
	// Calculate FPS for output
	calculateFPS();
	glutWarpPointer((glutGet(GLUT_WINDOW_WIDTH) / 2), (glutGet(GLUT_WINDOW_HEIGHT) / 2));	//Put mouse to middle of screen
	float mouseXOffset = input->getMouseX() - (glutGet(GLUT_WINDOW_WIDTH) / 2);		//
	float mouseYOffset = input->getMouseY() - (glutGet(GLUT_WINDOW_HEIGHT) / 2);	//		Offset from centre
	sceneCamera->RotateFromMouse(mouseXOffset, mouseYOffset, dt);	//Rotate based on offset
	sceneCamera->Animate(dt);	//Animate camera (if overridden to animate)
	lightManager.UpdateAll(dt);	//Update all lights (for flashing)
	slotMachine.Update(dt);	//Update slot machine (if activated)
}

//ResetValues sets the x, y and z values used for drawing the room planes back to their default values
void ResetValues(float &xMin, float &xMax, float &xIncr, float &yMin, float &yMax, float &yIncr, float &zMin, float &zMax, float &zIncr)
{
	xMin = 0.0f;
	xMax = 13.0f;
	xIncr = 0.25f;

	yMin = 0.0f;
	yMax = 4.0f;
	yIncr = 0.25f;

	zMin = -16.0f;
	zMax = 0.0f;
	zIncr = 0.25f;
}

//Draw Room Planes draws the 6 quad strips used to draw the room rectangle
void DrawRoomPlanes()
{
	float xMin = 0.0f;
	float xMax = 13.0f;
	float xIncr = 0.25f;

	float yMin = 0.0f;
	float yMax = 4.0f;
	float yIncr = 0.25f;

	float zMin = -16.0f;
	float zMax = 0.0f;
	float zIncr = 0.25f;

	//Floor
	glBindTexture(GL_TEXTURE_2D, floorTileTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	for (float xAxis = xMin; xAxis < xMax; xAxis += xIncr)
	{
		glBegin(GL_QUAD_STRIP);
		float xVal = (xAxis * 4.0f) * (24.0f / 52.0f);

		for (float zAxis = zMin; zAxis <= zMax; zAxis += zIncr)
		{
			float yVal = ((16 + zAxis) * 4.0f) * (24.0f / 64.0f);

			glNormal3f(0.0f, 1.0f, 0.0f);
			glTexCoord2f(xVal + (24.0f / 52.0f), yVal);
			glVertex3f(xAxis + xIncr, yMin, zAxis);
			glNormal3f(0.0f, 1.0f, 0.0f);
			glTexCoord2f(xVal, yVal);
			glVertex3f(xAxis, yMin, zAxis);
		}

		glEnd();
	}

	ResetValues(xMin, xMax, xIncr, yMin, yMax, yIncr, zMin, zMax, zIncr);

	//Roof
	glBindTexture(GL_TEXTURE_2D, roofTileTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	for (float xAxis = xMin; xAxis < xMax; xAxis += xIncr)
	{
		glBegin(GL_QUAD_STRIP);
		float xVal = (xAxis * 4.0f) * (8.0f / 52.0f);

		for (float zAxis = zMin; zAxis <= zMax; zAxis += zIncr)
		{
			float yVal = ((16 + zAxis) * 4.0f) * (8.0f / 64.0f);

			glNormal3f(0.0f, -1.0f, 0.0f);
			glTexCoord2f(xVal, yVal);
			glVertex3f(xAxis, yMax, zAxis);
			glNormal3f(0.0f, -1.0f, 0.0f);
			glTexCoord2f(xVal + (8.0f / 52.0f), yVal);
			glVertex3f(xAxis + xIncr, yMax, zAxis);
		}

		glEnd();
	}

	//Front Wall
	glBindTexture(GL_TEXTURE_2D, wallTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	for (float xAxis = xMin; xAxis < xMax; xAxis += xIncr)
	{
		glBegin(GL_QUAD_STRIP);
		float xVal = (xAxis * 4.0f) * (3.0f / 52.0f);

		for (float yAxis = yMin; yAxis <= yMax; yAxis += yIncr)
		{
			float yVal = 1 - ((yAxis * 4.0f) * (1.0f / 16.0f));

			glNormal3f(0.0f, 0.0f, -1.0f);
			glTexCoord2f(xVal, yVal);
			glVertex3f(xAxis + xIncr, yAxis, zMax);
			glNormal3f(0.0f, 0.0f, -1.0f);
			glTexCoord2f(xVal + (3.0f / 52.0f), yVal);
			glVertex3f(xAxis, yAxis, zMax);
		}

		glEnd();
	}

	ResetValues(xMin, xMax, xIncr, yMin, yMax, yIncr, zMin, zMax, zIncr);

	//Back Wall
	glBindTexture(GL_TEXTURE_2D, wallTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	for (float xAxis = xMin; xAxis < xMax; xAxis += xIncr)
	{
		glBegin(GL_QUAD_STRIP);
		float xVal = (xAxis * 4.0f) * (3.0f / 52.0f);

		for (float yAxis = yMin; yAxis <= yMax; yAxis += yIncr)
		{
			float yVal = 1 - ((yAxis * 4.0f) * (1.0f / 16.0f));

			glNormal3f(0.0f, 0.0f, 1.0f);
			glTexCoord2f(xVal, yVal);
			glVertex3f(xAxis, yAxis, zMin);
			glNormal3f(0.0f, 0.0f, 1.0f);
			glTexCoord2f(xVal + (3.0f / 52.0f), yVal);
			glVertex3f(xAxis + xIncr, yAxis, zMin);
		}

		glEnd();
	}

	ResetValues(xMin, xMax, xIncr, yMin, yMax, yIncr, zMin, zMax, zIncr);

	//Left Wall
	glBindTexture(GL_TEXTURE_2D, wallTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	for (float zAxis = zMax; zAxis > zMin; zAxis -= zIncr)
	{
		glBegin(GL_QUAD_STRIP);
		float xVal = (-zAxis * 4.0f) * (3.0f / 64.0f);

		for (float yAxis = yMin; yAxis <= yMax; yAxis += yIncr)
		{
			float yVal = 1 - ((yAxis * 4.0f) * (1.0f / 16.0f));

			glNormal3f(1.0f, 0.0f, 0.0f);
			glTexCoord2f(xVal, yVal);
			glVertex3f(xMin, yAxis, zAxis);
			glNormal3f(1.0f, 0.0f, 0.0f);
			glTexCoord2f(xVal + (3.0f / 64.0f), yVal);
			glVertex3f(xMin, yAxis, zAxis - zIncr);
		}

		glEnd();
	}

	ResetValues(xMin, xMax, xIncr, yMin, yMax, yIncr, zMin, zMax, zIncr);

	//Right Wall
	glBindTexture(GL_TEXTURE_2D, wallTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	for (float zAxis = zMax; zAxis > zMin; zAxis -= zIncr)
	{
		glBegin(GL_QUAD_STRIP);
		float xVal = (-zAxis * 4.0f) * (3.0f / 64.0f);

		for (float yAxis = yMin; yAxis <= yMax; yAxis += yIncr)
		{
			float yVal = 1 - ((yAxis * 4.0f) * (1.0f / 16.0f));

			glNormal3f(-1.0f, 0.0f, 0.0f);
			glTexCoord2f(xVal, yVal);
			glVertex3f(xMax, yAxis, zAxis - zIncr);
			glNormal3f(-1.0f, 0.0f, 0.0f);
			glTexCoord2f(xVal - (3.0f / 64.0f), yVal);
			glVertex3f(xMax, yAxis, zAxis);
		}

		glEnd();
	}

	ResetValues(xMin, xMax, xIncr, yMin, yMax, yIncr, zMin, zMax, zIncr);

}

//DrawCrate draws a square crate of size, starting at botLeftCorPos with texture
void DrawCrate(float size, Vector3 botLeftCorPos, GLuint texture, float maxU = 1.0f, float maxV = 1.0f)
{
	float minX = botLeftCorPos.x;
	float minY = botLeftCorPos.y;
	float minZ = botLeftCorPos.z;

	float maxX = botLeftCorPos.x + size;
	float maxY = botLeftCorPos.y + size;
	float maxZ = botLeftCorPos.z - size;

	glBindTexture(GL_TEXTURE_2D, texture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	//Front face
	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(minX, maxY, minZ);
		glTexCoord2f(0.0f, maxV);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(minX, minY, minZ);
		glTexCoord2f(maxU, maxV);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(maxX, minY, minZ);
		glTexCoord2f(maxU, 0.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(maxX, maxY, minZ);
	glEnd();

	//Back face
	glBegin(GL_QUADS);
		glTexCoord2f(maxU, 0.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(minX, maxY, maxZ);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(maxX, maxY, maxZ);
		glTexCoord2f(0.0f, maxV);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(maxX, minY, maxZ);
		glTexCoord2f(maxU, maxV);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(minX, minY, maxZ);
	glEnd();

	//Left face
	glBegin(GL_QUADS);
		glTexCoord2f(maxU, 0.0f);
		glNormal3f(-1.0f, 0.0f, 0.0f);
		glVertex3f(minX, maxY, minZ);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(-1.0f, 0.0f, 0.0f);
		glVertex3f(minX, maxY, maxZ);
		glTexCoord2f(0.0f, maxV);
		glNormal3f(-1.0f, 0.0f, 0.0f);
		glVertex3f(minX, minY, maxZ);
		glTexCoord2f(maxU, maxV);
		glNormal3f(-1.0f, 0.0f, 0.0f);
		glVertex3f(minX, minY, minZ);
	glEnd();

	//Right face
	glBegin(GL_QUADS);
		glTexCoord2f(maxU, 0.0f);
		glNormal3f(1.0f, 0.0f, 0.0f);
		glVertex3f(maxX, maxY, maxZ);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(1.0f, 0.0f, 0.0f);
		glVertex3f(maxX, maxY, minZ);
		glTexCoord2f(0.0f, maxV);
		glNormal3f(1.0f, 0.0f, 0.0f);
		glVertex3f(maxX, minY, minZ);
		glTexCoord2f(maxU, maxV);
		glNormal3f(1.0f, 0.0f, 0.0f);
		glVertex3f(maxX, minY, maxZ);
	glEnd();

	//Top face
	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(minX, maxY, maxZ);
		glTexCoord2f(maxU, 0.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(maxX, maxY, maxZ);
		glTexCoord2f(maxU, maxV);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(maxX, maxY, minZ);
		glTexCoord2f(0.0f, maxV);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(minX, maxY, minZ);
	glEnd();

	//Bottom face
	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, -1.0f, 0.0f);
		glVertex3f(minX, minY, minZ);
		glTexCoord2f(0.0f, maxV);
		glNormal3f(0.0f, -1.0f, 0.0f);
		glVertex3f(minX, minY, maxZ);
		glTexCoord2f(maxU, maxV);
		glNormal3f(0.0f, -1.0f, 0.0f);
		glVertex3f(maxX, minY, maxZ);
		glTexCoord2f(maxU, 0.0f);
		glNormal3f(0.0f, -1.0f, 0.0f);
		glVertex3f(maxX, minY, minZ);
	glEnd();

	glBindTexture(GL_TEXTURE_2D, NULL);
}

//DrawHandDrawnGeo draws any geometry made by hand, aka not a model or a procedurally generated shape
void DrawHandDrawnGeo()
{
	#pragma region Windows
	glBindTexture(GL_TEXTURE_2D, windowTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	glBegin(GL_QUAD_STRIP);
	for (float zAxis = -3.0f; zAxis >= -12.0f; zAxis -= 3.0f)
	{
		glTexCoord2f(zAxis / -3.0f, 0.0f);
		glNormal3f(1.0f, 0.0f, 0.0f);
		glVertex3f(0.1f, 3.0f, zAxis);
		glTexCoord2f(zAxis / -3.0f, 1.0f);
		glNormal3f(1.0f, 0.0f, 0.0f);
		glVertex3f(0.1f, 1.5f, zAxis);
	}
	glEnd();

	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 1.0f);
		glNormal3f(1.0f, 0.0f, -1.0f);
		glVertex3f(1.0f, 1.5f, -0.1f);

		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(1.0f, 3.0f, -0.1f);

		glTexCoord2f(1.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(4.0f, 3.0f, -0.1f);

		glTexCoord2f(1.0f, 1.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(4.0f, 1.5f, -0.1f);
	glEnd();

	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 1.0f);
		glNormal3f(1.0f, 0.0f, -1.0f);
		glVertex3f(9.0f, 1.5f, -0.1f);

		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(9.0f, 3.0f, -0.1f);

		glTexCoord2f(1.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(12.0f, 3.0f, -0.1f);

		glTexCoord2f(1.0f, 1.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(12.0f, 1.5f, -0.1f);
	glEnd();

	#pragma endregion

	#pragma region Flags
	glBindTexture(GL_TEXTURE_2D, flagTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	glBegin(GL_TRIANGLES);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(11.0f, 2.5f, -15.9f);
		glTexCoord2f(0.0f, 1.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(11.0f, 2.0f, -15.9f);
		glTexCoord2f(1.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(12.0f, 2.25f, -15.9f);
	glEnd();
	#pragma endregion

	#pragma region Chalkboard
		glBindTexture(GL_TEXTURE_2D, chalkboardTexture);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

		glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(6.0f, 2.5f, -15.9f);
		glTexCoord2f(0.0f, 1.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(6.0f, 1.75f, -15.9f);
		glTexCoord2f(1.0f, 1.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(6.75f, 1.75f, -15.9f);
		glTexCoord2f(1.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(6.75f, 2.5f, -15.9f);
		glEnd();


	#pragma endregion

	#pragma region TallPoster
		glBindTexture(GL_TEXTURE_2D, duffTallTexture);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

		glBegin(GL_QUADS);
			glTexCoord2f(0.0f, 0.0f);
			glNormal3f(0.0f, 0.0f, 1.0f);
			glVertex3f(2.0f, 3.0f, -15.9f);
			glTexCoord2f(0.0f, 1.0f);
			glNormal3f(0.0f, 0.0f, 1.0f);
			glVertex3f(2.0f, 1.5f, -15.9f);
			glTexCoord2f(1.0f, 1.0f);
			glNormal3f(0.0f, 0.0f, 1.0f);
			glVertex3f(3.0f, 1.5f, -15.9f);
			glTexCoord2f(1.0f, 0.0f);
			glNormal3f(0.0f, 0.0f, 1.0f);
			glVertex3f(3.0f, 3.0f, -15.9f);
		glEnd();
	#pragma endregion

	#pragma region WidePoster
		glBindTexture(GL_TEXTURE_2D, duffWideTexture);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

		glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(9.0f, 3.0f, -15.9f);
		glTexCoord2f(0.0f, 1.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(9.0f, 2.0f, -15.9f);
		glTexCoord2f(1.0f, 1.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(10.5f, 2.0f, -15.9f);
		glTexCoord2f(1.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, 1.0f);
		glVertex3f(10.5f, 3.0f, -15.9f);
		glEnd();
	#pragma endregion

	#pragma region DoorMat
		glBindTexture(GL_TEXTURE_2D, doorMatTexture);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

		glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(5.75f, 0.05f, -1.0f);
		glTexCoord2f(0.0f, 1.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(5.75f, 0.05f, -0.25f);
		glTexCoord2f(1.0f, 1.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(7.25f, 0.05f, -0.25f);
		glTexCoord2f(1.0f, 0.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(7.25f, 0.05f, -1.0f);
		glEnd();
	#pragma endregion

	#pragma region ShelfGlass
	glPushMatrix();
		glEnable(GL_BLEND);
		glDisable(GL_LIGHTING);

		glColor4f(0.3f, 0.3f, 0.9f, 0.6f);
		glBegin(GL_QUADS);
			glVertex3f(12.3f, 2.7f, -12.5f);
			glVertex3f(12.3f, 2.7f, -7.5f);
			glVertex3f(12.3f, 1.7f, -7.5f);
			glVertex3f(12.3f, 1.7f, -12.5f);
		glEnd();

		glEnable(GL_LIGHTING);
		glDisable(GL_BLEND);
	glPopMatrix();
	#pragma endregion

	#pragma region Door
	glBindTexture(GL_TEXTURE_2D, doorTexture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 1.0f);
		glNormal3f(1.0f, 0.0f, -1.0f);
		glVertex3f(5.75f, 0.0f, -0.1f);

		glTexCoord2f(0.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(5.75f, 3.0f, -0.1f);

		glTexCoord2f(1.0f, 0.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(7.25f, 3.0f, -0.1f);

		glTexCoord2f(1.0f, 1.0f);
		glNormal3f(0.0f, 0.0f, -1.0f);
		glVertex3f(7.25f, 0.0f, -0.1f);
	glEnd();
	#pragma endregion

	glBindTexture(GL_TEXTURE_2D, NULL);
}

//DrawModels renders all the loaded models into their position in the scene
void DrawModels()
{
	#pragma region Model Rendering
	poolTable.render(Vector3(6.0f, 0.5f, -9.5f));
	booth.render(Vector3(1.35f, 0.65f, -14.1f));
	booth.render(Vector3(1.35f, 0.65f, -10.6f));
	booth.render(Vector3(1.35f, 0.65f, -7.1f));
	bar.render(Vector3(10.75f, 0.75f, -9.5f));
	seat.render(Vector3(8.9f, 0.45f, -12.0f));
	seat.render(Vector3(9.1f, 0.45f, -10.5f));
	seat.render(Vector3(9.0f, 0.45f, -9.0f));
	seat.render(Vector3(8.9f, 0.45f, -7.5f));
	shelf.render(Vector3(12.4f, 1.5f, -10.0f));
	slotMachine.render(Vector3(12.4f, 1.1f, -2.0f));
	phone.render(Vector3(8.5f, 1.5f, -15.7f));
	jukebox.render(Vector3(6.5f, 0.65f, -15.4f));
	barrel.render(Vector3(12.1f, 0.4f, -13.6f));
	barrel.render(Vector3(11.4f, 0.4f, -13.6f));
	#pragma endregion

	//Glass Table
	glPushMatrix();
		glRotatef(90.0f, 0.0f, 1.0f, 0.0f);
		glassTable.render(Vector3(2.5f, 0.0f, 1.5f));
	glPopMatrix();

	#pragma region Torus Rendering
	glPushMatrix();
		glTranslatef(10.0f, 1.05f, -13.0f);
		glRotatef(-90.0f, 1.0f, 0.0f, 0.0f);
		donut.Draw();
	glPopMatrix();

	glPushMatrix();
		glTranslatef(10.2f, 1.05f, -12.9f);
		glRotatef(-90.0f, 1.0f, 0.0f, 0.0f);
		donut.Draw();
	glPopMatrix();

	glPushMatrix();
		glTranslatef(10.03f, 1.05f, -12.75f);
		glRotatef(-90.0f, 1.0f, 0.0f, 0.0f);
		donut.Draw();
	glPopMatrix();
	#pragma endregion
}

//DrawGlassTable draws the glass table itself, the reflected objects with the stencil and the transparent glass
void DrawGlassTable()
{
	//Stencil setup
	glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_ALWAYS, 1, 1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);

	glDisable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);

	glBegin(GL_QUADS);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(1.0f, 1.3f, -1.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(2.0f, 1.3f, -1.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(2.0f, 1.3f, -4.0f);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(1.0f, 1.3f, -4.0f);
	glEnd();

	glDisable(GL_CULL_FACE);

	//"Fake" reflected model
	glEnable(GL_DEPTH_TEST);
	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	glStencilFunc(GL_EQUAL, 1, 1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

	glPushMatrix();
		glScalef(0.2f, -0.2f, 0.2f);
		glTranslatef(0.0f, 1.45f, 0.0f);
		colaCan.render(Vector3(1.5f / 0.2f, 1.45f / -0.2f, -3.0f / 0.2f));
		colaCan.render(Vector3(1.5f / 0.2f, 1.45f / -0.2f, -2.5f / 0.2f));
		colaCan.render(Vector3(1.5f / 0.2f, 1.45f / -0.2f, -2.0f / 0.2f));
	glPopMatrix();

	glDisable(GL_STENCIL_TEST);

	glPushMatrix();
		glScalef(0.2f, 0.2f, 0.2f);
		colaCan.render(Vector3(1.5f / 0.2f, 1.45f / 0.2f, -3.0f / 0.2f));
		colaCan.render(Vector3(1.5f / 0.2f, 1.45f / 0.2f, -2.5f / 0.2f));
		colaCan.render(Vector3(1.5f / 0.2f, 1.45f / 0.2f, -2.0f / 0.2f));
	glPopMatrix();

	//Real model
	glEnable(GL_BLEND);
	glDisable(GL_LIGHTING);

	glPushMatrix();
		glColor4f(0.3f, 0.3f, 0.9f, 0.8f);
		glBegin(GL_QUADS);
			glNormal3f(0.0f, 1.0f, 0.0f);
			glVertex3f(1.0f, 1.3f, -1.0f);
			glNormal3f(0.0f, 1.0f, 0.0f);
			glVertex3f(2.0f, 1.3f, -1.0f);
			glNormal3f(0.0f, 1.0f, 0.0f);
			glVertex3f(2.0f, 1.3f, -4.0f);
			glNormal3f(0.0f, 1.0f, 0.0f);
			glVertex3f(1.0f, 1.3f, -4.0f);
		glEnd();

		glEnable(GL_LIGHTING);
		glDisable(GL_BLEND);
		
	glPopMatrix();
}

void Scene::render() {

	// Clear colour, depth and stencil buffers
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

	glLoadIdentity();
	gluLookAt(sceneCamera->GetPosition().x, sceneCamera->GetPosition().y, sceneCamera->GetPosition().z, sceneCamera->GetLookAt().x, sceneCamera->GetLookAt().y, sceneCamera->GetLookAt().z, sceneCamera->GetUp().x, sceneCamera->GetUp().y, sceneCamera->GetUp().z);

	// Render geometry/scene here -------------------------------------

	DrawRoomPlanes();
	DrawModels();
	DrawCrate(0.75f, Vector3(9.0f, 0.0f, -15.0f), crateTexture);
	DrawCrate(0.75f, Vector3(9.75f, 0.0f, -15.0f), crateTexture);
	DrawCrate(0.75f, Vector3(10.5f, 0.0f, -15.0f), crateTexture);
	DrawCrate(0.75f, Vector3(9.45f, 0.75f, -15.0f), crateTexture);
	DrawCrate(0.75f, Vector3(10.15f, 0.75f, -15.0f), crateTexture);
	DrawHandDrawnGeo();
	DrawGlassTable();

	lightManager.RenderAll();
	// End render geometry --------------------------------------

	// Render text, should be last object rendered.
	renderTextOutput();

	// Swap buffers, after all objects are rendered.
	glutSwapBuffers();
}

// Handles the resize of the window. If the window changes size the perspective matrix requires re-calculation to match new window size.
void Scene::resize(int w, int h)
{
	width = w;
	height = h;
	// Prevent a divide by zero, when window is too short
	// (you cant make a window of zero width).
	if (h == 0)
		h = 1;

	float ratio = (float)w / (float)h;
	fov = 45.0f;
	nearPlane = 0.1f;
	farPlane = 100.0f;

	// Use the Projection Matrix
	glMatrixMode(GL_PROJECTION);

	// Reset Matrix
	glLoadIdentity();

	// Set the viewport to be the entire window
	glViewport(0, 0, w, h);

	// Set the correct perspective.
	gluPerspective(fov, ratio, nearPlane, farPlane);

	// Get Back to the Modelview
	glMatrixMode(GL_MODELVIEW);


}

// Calculates FPS
void Scene::calculateFPS()
{
	frame++;
	time = glutGet(GLUT_ELAPSED_TIME);

	if (time - timebase > 1000) {
		sprintf_s(fps, "FPS: %4.2f", frame*1000.0 / (time - timebase));
		timebase = time;
		frame = 0;
	}
}

// Compiles standard output text including FPS and current mouse position.
void Scene::renderTextOutput()
{
	// Render current mouse position and frames per second.
	sprintf_s(mouseText, "Mouse: %i, %i", input->getMouseX(), input->getMouseY());
	sprintf_s(posText, "Camera Pos: %4.2f, %4.2f, %4.2f", sceneCamera->GetLookAt().x, sceneCamera->GetLookAt().y, sceneCamera->GetLookAt().z);
	sprintf_s(cameraText, "PitchYawRoll: %4.2f, %4.2f, %4.2f", sceneCamera->GetPitchYawRoll().x, sceneCamera->GetPitchYawRoll().y, sceneCamera->GetPitchYawRoll().z);
	/*displayText(-1.f, 0.96f, 1.f, 0.f, 0.f, mouseText);
	displayText(-1.f, 0.90f, 1.f, 0.f, 0.f, fps);
	displayText(-1.f, 0.84f, 1.f, 0.f, 0.f, posText);
	displayText(-1.f, 0.78f, 1.f, 0.f, 0.f, cameraText);*/
}

// Renders text to screen. Must be called last in render function (before swap buffers)
void Scene::displayText(float x, float y, float r, float g, float b, char* string) {
	// Get Lenth of string
	int j = strlen(string);

	// Swap to 2D rendering
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(-1.0, 1.0, -1.0, 1.0, 5, 100);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	// Orthographic lookAt (along the z-axis).
	gluLookAt(0.0f, 0.0f, 10.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);

	// Set text colour and position.
	glColor3f(r, g, b);
	glRasterPos2f(x, y);
	// Render text.
	for (int i = 0; i < j; i++) {
		glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, string[i]);
	}
	// Reset colour to white.
	glColor3f(1.f, 1.f, 1.f);

	// Swap back to 3D rendering.
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(fov, ((float)width / (float)height), nearPlane, farPlane);
	glMatrixMode(GL_MODELVIEW);
}
